#include "ising.h"
#include "statistics.h"


void Ising::cold_start() //all spins 1
{
    int num;
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    int size = sqrt(num);
    

    for(unsigned int i = 0; i < m_Nx/size+2; i++)
    {
        for(unsigned int j = 0; j < m_Ny/size+2; j++)
        {
	    mgrid[(m_Nx/size+2)*j+i] = 1;
        }
    }

    MPI_Barrier(MPI_COMM_WORLD);
}

void Ising::hot_start() //all spins random
{
    int num;
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    int size = sqrt(num);

    std::random_device device;
    std::mt19937 generator(device());


    for(unsigned int i = 0; i < m_Nx/size+2; i++)
    {
        for(unsigned int j = 0; j < m_Ny/size+2; j++)
        {
	    std::uniform_int_distribution<int> distribution(0,1);
            if(distribution(generator)%2 == 1) mgrid[(m_Nx/size+2)*j+i] =  1;
            else                               mgrid[(m_Nx/size+2)*j+i] = -1;
        }
    }

    MPI_Barrier(MPI_COMM_WORLD);
}

double Ising::compute_energy() //store initial value for energy
{
    int num;
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    int size = sqrt(num);
    
    int spin_sum = 0;
    int neighbour_sum = 0;
    for(unsigned int i = 1; i <= m_Nx/size; i++)
    {
        for(unsigned int j = 1; j <= m_Ny/size; j++)
	{
	    int point = (m_Nx/size+2)*j+i;
	    
            spin_sum += mgrid[point];
	    
            neighbour_sum += mgrid[point]*mgrid[point+1];
            neighbour_sum += mgrid[point]*mgrid[point+(m_Nx/size+2)];
	}
    }
    
    return -mJ*neighbour_sum - mB*spin_sum;
}

double Ising::delta_energy(int i, int j)
{
    int num;
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    int size = sqrt(num);

    int point = (m_Nx/size+2)*j+i;
    int spin_sum = mgrid[point];
    int neighbour_sum = 0;
 
    neighbour_sum += mgrid[point]*mgrid[point-1]; //mid point
    neighbour_sum += mgrid[point]*mgrid[point+1];
    neighbour_sum += mgrid[point]*mgrid[point-(m_Nx/size+2)];
    neighbour_sum += mgrid[point]*mgrid[point+(m_Nx/size+2)];

    return 2.0*mJ*neighbour_sum + mB*spin_sum;
}

void Ising::MC_sweep() //metropolis algorithm
{
    int num;
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    int size = sqrt(num);
    
    std::random_device device;
    std::mt19937 generator(device());
       
    double dE, randn, p;
    for(unsigned int i = 1; i <= m_Ny/size; i++)
    {
        for(unsigned int j = 1; j <= m_Nx/size; j++)
        {	    
	    dE = delta_energy(i,j);
            p = exp(-m_beta*dE);
	  
            std::uniform_int_distribution<int> distribution(0,1000000000);
            randn = distribution(generator)/1000000000.0;

            if(dE < 0 || p > randn)
            {
	        mgrid[(m_Nx/size+2)*j+i] *= -1;

                mE += dE;
            }
        }
    }
}

void Ising::production_run(unsigned int steps)
{
    int rank, num;
    MPI_Comm_size(MPI_COMM_WORLD, &num );
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    int size = sqrt(num);
    MPI_Status* status;

    energies.resize(0);
    double E = 0.0;
    unsigned int V = m_Nx*m_Ny;
    
    int rv[m_Ny/size];
    int lv[m_Ny/size];
    int uv[m_Nx/size];
    int dv[m_Nx/size];

    int rc_i = rank/size;
    int rc_j = rank%size;

    int ln   = rc_i*size + (rc_j+size-1) %size;
    int rn   = rc_i*size + (rc_j+size+1) %size;
    int upn  = (rc_i+size-1)%size*size + rc_j;
    int lown = (rc_i+size+1)%size*size + rc_j;
    
    
    for(unsigned int k = 0; k < steps; k++)
    {
	//edges
	for(unsigned int i = 0; i < m_Nx/size; i++)
	{
	  //std::cout << i << " " << m_Ny/size*(m_Nx/size+1)+m_Ny/size+1+i << std::endl;
	    dv[i] = mgrid[m_Nx/size*(m_Ny/size+2)+1+i];
	}
	for(unsigned int j = 0; j < m_Ny/size; j++)
	{
	  //std::cout << j << " " << (j+2)*(m_Nx/size+1)+j << std::endl;
	    rv[j] = mgrid[(j+2)*(m_Nx/size+1)+j];
	}
	  
	//edge communication
	if(rc_i == rc_j)
	{
	    MPI_Send(dv, m_Nx/size, MPI_INT, lown, 0, MPI_COMM_WORLD);
	    MPI_Recv(uv, m_Nx/size, MPI_INT,  upn, 0, MPI_COMM_WORLD, status);
	    
	    //std::cout << rank << ":" << lown << std::endl;

	    MPI_Send(rv, m_Ny/size, MPI_INT, rn, 1, MPI_COMM_WORLD);
	    MPI_Recv(lv, m_Ny/size, MPI_INT, ln, 1, MPI_COMM_WORLD, status);

	    //std::cout << rank << ":" << rn << std::endl;
	}
	else
	{
	    MPI_Recv(uv, m_Nx/size, MPI_INT,  upn, 0, MPI_COMM_WORLD, status);
	    MPI_Send(dv, m_Nx/size, MPI_INT, lown, 0, MPI_COMM_WORLD);
	      
            //std::cout << rank << ":" << lown << std::endl;
	      
	    MPI_Recv(lv, m_Ny/size, MPI_INT, ln, 1, MPI_COMM_WORLD,  status);
	    MPI_Send(rv, m_Ny/size, MPI_INT, rn, 1, MPI_COMM_WORLD);

	    //std::cout << rank << ":" << ln << std::endl;
	}

	//update overlap
	for(unsigned int i = 0; i < m_Nx/size; i++)
	{
	    mgrid[i+1] = uv[i];
	}
	for(unsigned int j = 0; j < m_Ny/size; j++)
	{
	    mgrid[(m_Nx/size+2)*(j+1)] = lv[j];
	}
	
	//////////////////////////////////////////////////////////////////////////////
	  
	//edges
        for(unsigned int i = 0; i < m_Nx/size; i++)
        {
	    uv[i] = mgrid[m_Nx/size+3+i];
        }
	for(unsigned int j = 0; j < m_Ny/size; j++)
	{
	    lv[j] = mgrid[(m_Nx/size+2)*(j+1)+1];
	}

	//edge communication
	if(rc_i == rc_j)
	{
            MPI_Send(uv, m_Nx/size, MPI_INT,  upn, 2, MPI_COMM_WORLD);
	    MPI_Recv(dv, m_Nx/size, MPI_INT, lown, 2, MPI_COMM_WORLD, status);
	
            MPI_Send(lv, m_Ny/size, MPI_INT, ln, 3, MPI_COMM_WORLD);
	    MPI_Recv(rv, m_Ny/size, MPI_INT, rn, 3, MPI_COMM_WORLD, status);
	}
	else
	{
	    MPI_Recv(dv, m_Nx/size, MPI_INT, lown, 2, MPI_COMM_WORLD, status);
	    MPI_Send(uv, m_Nx/size, MPI_INT,  upn, 2, MPI_COMM_WORLD);

	    MPI_Recv(rv, m_Ny/size, MPI_INT, rn, 3, MPI_COMM_WORLD, status);
            MPI_Send(lv, m_Ny/size, MPI_INT, ln, 3, MPI_COMM_WORLD);
	}

	//update overlap
	for(unsigned int i = 0; i < m_Nx/size; i++)
	{
	    mgrid[(m_Nx/size+1)*(m_Ny/size+2)+1+i] = dv[i];
	}
	for(unsigned int j = 0; j < m_Ny/size; j++)
	{
	    mgrid[(j+2)*(m_Nx/size+1)+j+1] = rv[j];
	}

	
        if(k == 0) mE = this->compute_energy();
	MPI_Barrier(MPI_COMM_WORLD);
	
	this->MC_sweep();
	MPI_Barrier(MPI_COMM_WORLD);
        mE = this->compute_energy();
        /////////////////////////////////////////////////////////////////////////////

	//compute and store energy
        MPI_Reduce(&mE, &E, 1, MPI_DOUBLE, MPI_SUM, 0, MPI_COMM_WORLD);
        if(rank == 0)
	{
	  energies.push_back(E/V);
	  //std::cout << E/V << std::endl;
	}
    }
}
