#include <cstdlib>
#include <vector>
#include <random>
#include <cmath>
#include <iostream>
#include <mpi.h>
#include "ising.h"
#include "statistics.h"

int main(int argc, char **argv)
{
    MPI_Init(&argc, &argv);

    if(argc < 8 || argc > 9)
    {
      std::cerr << "Usage: ising <number of spins in x-direction> <number of spins in y-direction> <initial sate: hot_start/cold_start> <beta> <J> <B> <steps> <optional: # of simulations>" << std::endl;
      MPI_Abort(MPI_COMM_WORLD, 1);
    }
    
    int rank, num, size;
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    MPI_Comm_size(MPI_COMM_WORLD, &num);
    size = d2i(sqrt(num));
    if(size*size != num)
    {
      std::cerr << "The Algorithm requires a quadratic number of processes." << std::endl;
      MPI_Abort(MPI_COMM_WORLD, 2);
    }

    
    unsigned int Nx = atoi(argv[1]);
    unsigned int Ny = atoi(argv[2]);
    if(Nx%size != 0 || Ny%size != 0)
    {
      std::cerr << "The Algorithm requires sizes, which are a multiple of the square root of the process number to avoid unused overlap." << std::endl;
      MPI_Abort(MPI_COMM_WORLD, 3);
    }
    std::string hc = argv[3];
    double beta = atof(argv[4]);
    double J = atof(argv[5]);
    double B = atof(argv[6]);
    int steps = atoi(argv[7]);
    int sim_number;
    if(argv[8] != NULL) sim_number = atoi(argv[8]);
    else sim_number = 1;

    std::vector<double> time;

    std::vector<double> energies_uncorr;
    double mean_e, error_e, mean_time, error_time;
    
  double wtime = 0.0;
  for(int i = 0; i < sim_number; i++)
  {  
    wtime = MPI_Wtime();
    Ising *I = new Ising(beta, B, J, Nx, Ny, hc, size);
    I->production_run(steps);
    wtime = MPI_Wtime()-wtime;
    
    if(rank == 0)
    {
      time.push_back(wtime);
      
      calculate_observable(I->energies, energies_uncorr, mean_e, error_e);
      std::cout << "Energy: " << mean_e << " +-" << error_e << std::endl;
    }
  }

  if(rank == 0)
  {
    mean_time = mean(time);
    error_time = cov_func(0,time); //standard deviation
    std::cout << Nx << " " << mean_time << " " << error_time << std::endl;
  }
  
  MPI_Finalize();

  
  return 0;
}
